using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;
using Microsoft.Xna.Framework.Net;
using Microsoft.Xna.Framework.Storage;

using System.Threading.Tasks; //TPL

namespace Stereoscopy
{
    /// <summary>
    /// This is the main type for your game
    /// </summary>
    public class Game1 : Microsoft.Xna.Framework.Game
    {
        GraphicsDeviceManager graphics;
        SpriteBatch spriteBatch;

        GraphicsDevice gd;
        BasicEffect effect;

        Matrix viewLeft = Matrix.Identity;
        Matrix viewRight = Matrix.Identity;

        RenderTarget2D renderTargetLeft;
        RenderTarget2D renderTargetRight;
        
        //Red-Blue
        //Vector3 filterLeft = new Vector3(1, 0, 0);
        //Vector3 filterRight = new Vector3(0, 0, 1);

        //Red-Cyan
        Vector3 filterLeft = new Vector3(1, 0, 0);
        Vector3 filterRight = new Vector3(0, 1, 1);

        //Bursztyn-Granat (ColorCode 3D)
        //Vector3 filterLeft = new Vector3(0.85f, 0.85f, 0.3f);
        //Vector3 filterRight = new Vector3(0.15f, 0.15f, 0.7f);

        BigCube solid;
        bool isRotating = true;

        bool cameraEnabledLeft = true;
        bool cameraEnabledRight = true;
        bool showSeparateViews = true;

        readonly Vector3 cameraDefaultPositionLeft = new Vector3(0.92f, 1f, 4.5f);
        readonly Vector3 cameraDefaultPositionRight = new Vector3(1.08f, 1f, 4.5f);

        Vector3 cameraPositionLeft;
        Vector3 cameraPositionRight;

        enum SuperimposingMethod { OnlyCPU, CPUandGPU };
        SuperimposingMethod superimposingMethod = SuperimposingMethod.CPUandGPU;

        Effect superimposingEffect;

        public Game1()
        {
            graphics = new GraphicsDeviceManager(this);
            Content.RootDirectory = "Content";
        }
       
        protected override void Initialize()
        {
            gd = graphics.GraphicsDevice;
            PresentationParameters pp = gd.PresentationParameters;

            renderTargetLeft = new RenderTarget2D(gd, pp.BackBufferWidth, pp.BackBufferHeight, false, gd.DisplayMode.Format, DepthFormat.Depth24);
            renderTargetRight = new RenderTarget2D(gd, pp.BackBufferWidth, pp.BackBufferHeight, false, gd.DisplayMode.Format, DepthFormat.Depth24);

            effect = new BasicEffect(gd);
            effect.VertexColorEnabled = false;
            effect.Projection = Matrix.CreatePerspective(2.0f * gd.Viewport.AspectRatio, 2.0f, 1.0f, 100.0f);
            effect.EnableDefaultLighting();

            cameraPositionLeft = cameraDefaultPositionLeft;
            cameraPositionRight = cameraDefaultPositionRight;

            viewLeft = Matrix.CreateLookAt(cameraPositionLeft, Vector3.Zero, Vector3.Up);
            viewRight = Matrix.CreateLookAt(cameraPositionRight, Vector3.Zero, Vector3.Up);

            base.Initialize();
        }

       
        protected override void LoadContent()
        {
            superimposingEffect = Content.Load<Effect>("StereoscopyEffect");
            solid = new BigCube(gd, effect, Content.Load<Texture2D>("CubeTexture"));
            solid.SetXShape();
            spriteBatch = new SpriteBatch(gd);

            float kat=2.65f;
            solid.TransformWorld(Matrix.CreateRotationX(kat) * Matrix.CreateRotationY(kat));
        }

       
        protected override void UnloadContent()
        {
        }

        KeyboardState previousKeyboardState = Keyboard.GetState();

        protected override void Update(GameTime gameTime)
        {
           
            if (GamePad.GetState(PlayerIndex.One).Buttons.Back == ButtonState.Pressed)
                this.Exit();

            //keyboard
            KeyboardState keyboardState = Keyboard.GetState();
            if (keyboardState.IsKeyDown(Keys.Escape)) this.Exit();
            if (keyboardState.IsKeyDown(Keys.Space) && !previousKeyboardState.IsKeyDown(Keys.Space))
            {
                try
                {
                    graphics.ToggleFullScreen();
                }
                catch
                {                    
                }
            }

            float verticalRotation = 0.0f;
            float horizontalRotation = 0.0f;
            if (keyboardState.IsKeyDown(Keys.Up)) verticalRotation = -0.02f;
            if (keyboardState.IsKeyDown(Keys.Down)) verticalRotation = 0.02f;
            if (keyboardState.IsKeyDown(Keys.Right)) horizontalRotation = 0.02f;
            if (keyboardState.IsKeyDown(Keys.Left)) horizontalRotation = -0.02f;
            Quaternion aditionalRotation = Quaternion.CreateFromAxisAngle(new Vector3(1, 0, 0), verticalRotation) *
                                           Quaternion.CreateFromAxisAngle(new Vector3(0, 1, 0), horizontalRotation);

            if (keyboardState.IsKeyDown(Keys.F1)) solid.SetStarShape();
            if (keyboardState.IsKeyDown(Keys.F2)) solid.SetStairsShape();
            if (keyboardState.IsKeyDown(Keys.F3)) solid.SetXShape();
            if (keyboardState.IsKeyDown(Keys.F4)) solid.SetPShape();
            if (keyboardState.IsKeyDown(Keys.F5)) solid.SetOShape();
            if (keyboardState.IsKeyDown(Keys.F6)) solid.SetTShape();
            if (keyboardState.IsKeyDown(Keys.F7)) solid.SetCubeShape();

            if (keyboardState.IsKeyDown(Keys.L) && !previousKeyboardState.IsKeyDown(Keys.L)) cameraEnabledLeft = !cameraEnabledLeft;
            if (keyboardState.IsKeyDown(Keys.R) && !previousKeyboardState.IsKeyDown(Keys.R)) cameraEnabledRight = !cameraEnabledRight;
            if (keyboardState.IsKeyDown(Keys.D0) && !previousKeyboardState.IsKeyDown(Keys.D0)) isRotating = !isRotating;
            if (keyboardState.IsKeyDown(Keys.T) && !previousKeyboardState.IsKeyDown(Keys.T)) showSeparateViews = !showSeparateViews;
            if (keyboardState.IsKeyDown(Keys.D1)) superimposingMethod = SuperimposingMethod.OnlyCPU;
            if (keyboardState.IsKeyDown(Keys.D2)) superimposingMethod = SuperimposingMethod.CPUandGPU;

            if (keyboardState.IsKeyDown(Keys.PageDown))
            {
                cameraPositionLeft.X += 0.001f;
                cameraPositionRight.X -= 0.001f;

                viewLeft = Matrix.CreateLookAt(cameraPositionLeft, Vector3.Zero, Vector3.Up);
                viewRight = Matrix.CreateLookAt(cameraPositionRight, Vector3.Zero, Vector3.Up);
            }

            if (keyboardState.IsKeyDown(Keys.PageUp))
            {
                cameraPositionLeft.X -= 0.001f;
                cameraPositionRight.X += 0.001f;

                viewLeft = Matrix.CreateLookAt(cameraPositionLeft, Vector3.Zero, Vector3.Up);
                viewRight = Matrix.CreateLookAt(cameraPositionRight, Vector3.Zero, Vector3.Up);
            }

            if (keyboardState.IsKeyDown(Keys.Home))
            {
                cameraPositionLeft = cameraDefaultPositionLeft;
                cameraPositionRight = cameraDefaultPositionRight;

                viewLeft = Matrix.CreateLookAt(cameraPositionLeft, Vector3.Zero, Vector3.Up);
                viewRight = Matrix.CreateLookAt(cameraPositionRight, Vector3.Zero, Vector3.Up);
            }

            if (keyboardState.IsKeyDown(Keys.End))
            {
                cameraPositionLeft = (cameraDefaultPositionLeft + cameraDefaultPositionRight) / 2;
                cameraPositionRight = cameraPositionLeft;

                viewLeft = Matrix.CreateLookAt(cameraPositionLeft, Vector3.Zero, Vector3.Up);
                viewRight = Matrix.CreateLookAt(cameraPositionRight, Vector3.Zero, Vector3.Up);
            }

            previousKeyboardState = keyboardState;
                
            Matrix transformation;
            if (!isRotating)
            {
                transformation = Matrix.CreateFromQuaternion(aditionalRotation);
            }
            else
            {
                transformation = Matrix.CreateRotationX(gameTime.ElapsedGameTime.Milliseconds / 1000.0f)
                   * Matrix.CreateRotationY(gameTime.ElapsedGameTime.Milliseconds / 1000.0f);
            }
            if (keyboardState.IsKeyDown(Keys.OemPlus)) transformation *= Matrix.CreateScale(1.01f);
            if (keyboardState.IsKeyDown(Keys.OemMinus)) transformation *= Matrix.CreateScale(0.99f);
            solid.TransformWorld(transformation);
           
            base.Update(gameTime);
        }

        private string Description(GameTime gameTime)
        {
            string description = "Stereoscopy - ";
            switch (superimposingMethod)
            {
                case SuperimposingMethod.OnlyCPU: description += "CPU only, "; break;
                case SuperimposingMethod.CPUandGPU: description += "CPU & GPU, "; break;
            }
            if (gameTime.IsRunningSlowly) description += " (runs slowly), ";
            description += "cameras distance: " + (cameraPositionRight.X - cameraPositionLeft.X).ToString();
            description += ", keys: 0,arrow keys,1-2,F1-F7,R,L,T,Home,End,PgUp,PgDown";
            //Window.Title = description;
            return description;
        }

        protected override void Draw(GameTime gameTime)
        {
            Window.Title = Description(gameTime);
            switch (superimposingMethod)
            {
                case SuperimposingMethod.OnlyCPU: Draw0(gameTime); break;
                case SuperimposingMethod.CPUandGPU: Draw1(gameTime); break;
            }
        }

        //CPU only
        protected void Draw0(GameTime gameTime)
        {
            //Left camera
            gd.SetRenderTarget(renderTargetLeft);
            gd.Clear(Color.Black);
            if (cameraEnabledLeft)
            {
                solid.View = viewLeft;
                solid.Draw();
            }
            gd.SetRenderTarget(null);

            //Right camera
            gd.SetRenderTarget(renderTargetRight);
            gd.Clear(Color.Black);
            if (cameraEnabledRight)
            {
                solid.View = viewRight;
                solid.Draw();
            }
            gd.SetRenderTarget(null);

            //Superimposing
            Color[] imageLeft = new Color[renderTargetLeft.Width * renderTargetLeft.Height];
            renderTargetLeft.GetData<Color>(imageLeft);
            Color[] imageRight = new Color[renderTargetRight.Width * renderTargetRight.Height];
            renderTargetRight.GetData<Color>(imageRight);
            //for (int i = 0; i < imageRight.Count(); ++i) imageRight[i].R = imageLeft[i].R;
            //Parallel.For(0, imageRight.Count(),  i => { imageRight[i].R = imageLeft[i].R; });
            //for (int i = 0; i < imageRight.Count(); ++i) imageRight[i] = new Color(imageLeft[i].ToVector3() * filterLeft + imageRight[i].ToVector3() * filterRight);
            Parallel.For(0, imageRight.Count(), i => { imageRight[i] = new Color(imageLeft[i].ToVector3() * filterLeft + imageRight[i].ToVector3() * filterRight); });
            Texture2D zlozenie3D = new Texture2D(gd, renderTargetRight.Width, renderTargetRight.Height);
            zlozenie3D.SetData<Color>(imageRight);

            spriteBatch.Begin(SpriteSortMode.Immediate, BlendState.Opaque, null, DepthStencilState.Default, null);
            spriteBatch.Draw(zlozenie3D, new Rectangle(0, 0, Window.ClientBounds.Width, Window.ClientBounds.Height), Color.White);
            if (showSeparateViews)
            {
                spriteBatch.Draw(renderTargetLeft, new Rectangle(0, 0, Window.ClientBounds.Width / 4, Window.ClientBounds.Height / 4), Color.White);
                spriteBatch.Draw(renderTargetRight, new Rectangle(3 * Window.ClientBounds.Width / 4, 0, Window.ClientBounds.Width / 4, Window.ClientBounds.Height / 4), Color.White);
            }
            spriteBatch.End();

            base.Draw(gameTime);
        }

        VertexPositionTexture[] werteksyEkranu = new VertexPositionTexture[4]
        {
                new VertexPositionTexture(new Vector3(-1, 1, 0f), new Vector2(0, 0)),
                new VertexPositionTexture(new Vector3(1, 1, 0f), new Vector2(1, 0)),
                new VertexPositionTexture(new Vector3(-1, -1, 0f), new Vector2(0, 1)),
                new VertexPositionTexture(new Vector3(1, -1, 0f), new Vector2(1, 1))
        };

        //CPU & GPU
        protected void Draw1(GameTime gameTime)
        {
            //Left camera
            gd.SetRenderTarget(renderTargetLeft);
            gd.Clear(Color.Black);
            if (cameraEnabledLeft)
            {
                solid.View = viewLeft;
                solid.Draw();
            }
            gd.SetRenderTarget(null);

            //Right camera
            gd.SetRenderTarget(renderTargetRight);
            gd.Clear(Color.Black);
            if (cameraEnabledRight)
            {
                solid.View = viewRight;
                solid.Draw();
            }
            gd.SetRenderTarget(null);

            //Superimposing
            superimposingEffect.CurrentTechnique = superimposingEffect.Techniques[0];
            superimposingEffect.Parameters["TextureLeft"].SetValue(renderTargetLeft);
            superimposingEffect.Parameters["TextureRight"].SetValue(renderTargetRight);
            superimposingEffect.Parameters["FilterLeft"].SetValue(filterLeft);
            superimposingEffect.Parameters["FilterRight"].SetValue(filterRight);

            foreach (EffectPass pass in superimposingEffect.CurrentTechnique.Passes)
            {
                pass.Apply();
                gd.DrawUserPrimitives<VertexPositionTexture>(PrimitiveType.TriangleStrip, werteksyEkranu, 0, 2);
            }

            if (showSeparateViews)
            {
                spriteBatch.Begin(SpriteSortMode.Immediate, BlendState.Opaque, null, DepthStencilState.Default, null);
                //spriteBatch.Draw(renderTargetRight, new Rectangle(0, 0, Window.ClientBounds.Width, Window.ClientBounds.Height), Color.White);
                spriteBatch.Draw(renderTargetLeft, new Rectangle(0, 0, Window.ClientBounds.Width / 4, Window.ClientBounds.Height / 4), Color.White);
                spriteBatch.Draw(renderTargetRight, new Rectangle(3 * Window.ClientBounds.Width / 4, 0, Window.ClientBounds.Width / 4, Window.ClientBounds.Height / 4), Color.White);
                spriteBatch.End();
            }

            base.Draw(gameTime);
        }
    }
}
